<?php

namespace App\Support;

class DateHelper
{
    public static function normalizeDigits(string $value): string
    {
        $persian = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹'];
        $arabic  = ['٠','١','٢','٣','٤','٥','٦','٧','٨','٩'];
        $english = ['0','1','2','3','4','5','6','7','8','9'];
        $value = str_replace($persian, $english, $value);
        return str_replace($arabic, $english, $value);
    }

    public static function isJalaliDate(string $date): bool
    {
        $parts = explode('-', $date);
        if (count($parts) !== 3) return false;
        $year = intval($parts[0]);
        return $year >= 1300 && $year <= 1499;
    }

    public static function jalaliToGregorian(int $jy, int $jm, int $jd): array
    {
        $jy += 1595;
        $days = -355668 + (365 * $jy) + (int)($jy / 33) * 8 + (int)(($jy % 33 + 3) / 4) + $jd + (($jm < 7) ? ($jm - 1) * 31 : (($jm - 7) * 30) + 186);
        $gy = 400 * (int)($days / 146097);
        $days %= 146097;
        if ($days > 36524) {
            $days--;
            $gy += 100 * (int)($days / 36524);
            $days %= 36524;
            if ($days >= 365) $days++;
        }
        $gy += 4 * (int)($days / 1461);
        $days %= 1461;
        if ($days > 365) {
            $gy += (int)(($days - 1) / 365);
            $days = ($days - 1) % 365;
        }
        $gd = $days + 1;
        $sal_a = [0, 31, (($gy % 4 === 0 && $gy % 100 !== 0) || ($gy % 400 === 0)) ? 29 : 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        for ($gm = 0; $gm < 13; $gm++) {
            $v = $sal_a[$gm];
            if ($gd <= $v) break;
            $gd -= $v;
        }
        return [$gy, $gm, $gd];
    }

    public static function toGregorian(string $date): string
    {
        $date = self::normalizeDigits($date);
        if (!self::isJalaliDate($date)) {
            return $date;
        }
        [$jy, $jm, $jd] = array_map('intval', explode('-', $date));
        [$gy, $gm, $gd] = self::jalaliToGregorian($jy, $jm, $jd);
        return sprintf('%04d-%02d-%02d', $gy, $gm, $gd);
    }
}
