<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use Illuminate\Http\Request;
use App\Support\DateHelper;

class BookingController extends Controller
{
    public function index(Request $request)
    {
        // Manager can see all, Customer sees their own
        $user = $request->user();
        if ($user->role === 'MANAGER') {
            return response()->json(Booking::with(['staff', 'services'])->orderBy('created_at', 'desc')->get());
        }
        if ($user->role === 'STAFF') {
            $staff = \App\Models\Staff::where('phone', $user->phone)->first();
            if ($staff) {
                return response()->json(Booking::with(['staff', 'services'])->where('staff_id', $staff->id)->orderBy('created_at', 'desc')->get());
            }
            return response()->json([]);
        }
        return response()->json(Booking::with(['staff', 'services'])->where('customer_phone', $user->phone)->orderBy('created_at', 'desc')->get());
    }

    public function store(Request $request)
    {
        if ($request->has('date')) {
            $request->merge(['date' => DateHelper::toGregorian($request->input('date'))]);
        }
        $validated = $request->validate([
            'id' => 'required|unique:bookings,id',
            'staff_id' => 'required',
            'date' => 'required|date_format:Y-m-d',
            'time' => 'required',
            'payment_type' => 'required',
            'amount_paid' => 'required',
            'customer_phone' => 'required',
            'customer_name' => 'nullable|string',
            'service_ids' => 'required|array'
        ]);

        $booking = Booking::create($request->except('service_ids'));

        if ($request->has('service_ids')) {
            $booking->services()->attach($request->service_ids);
        }

        return response()->json($booking->load(['staff', 'services']), 201);
    }

    public function update(Request $request, $id)
    {
        if ($request->has('date')) {
            $request->merge(['date' => DateHelper::toGregorian($request->input('date'))]);
        }
        $booking = Booking::findOrFail($id);
        $booking->update($request->except('service_ids'));
        
        $request->validate([
            'customer_name' => 'nullable|string',
            'date' => 'sometimes|date_format:Y-m-d',
        ]);        
        if ($request->has('service_ids')) {
            $booking->services()->sync($request->service_ids);
        }
        
        return response()->json($booking->load(['staff', 'services']));
    }

    public function destroy($id)
    {
        Booking::findOrFail($id)->delete();
        return response()->json(null, 204);
    }
}
