<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class AuthController extends Controller
{
    public function login(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'password' => 'nullable|string',
        ]);

        $staff = \App\Models\Staff::where('phone', $request->phone)->first();

        // If this phone belongs to a staff member, enforce password-based login before any user creation.
        if ($staff && !$request->password) {
            return response()->json(['message' => 'Password is required for staff login'], 422);
        }

        $user = User::where('phone', $request->phone)->first();

        if (!$user) {
            $user = User::create([
                'phone' => $request->phone,
                'name' => $staff ? $staff->name : 'کاربر مهمان',
                'role' => $staff ? 'STAFF' : 'CUSTOMER',
                'password' => Hash::make($staff ? ($request->password ?? $request->phone) : $request->phone),
            ]);
        } else {
            // Update role if user is now a staff member but recorded as customer
            if ($staff && $user->role !== 'STAFF' && $user->role !== 'MANAGER') {
                $user->role = 'STAFF';
                $user->name = $staff->name;
                $user->save();
            }
        }

        // If this phone belongs to a staff member, enforce password-based login
        if ($staff) {
            if (!$user->password || !Hash::check($request->password, $user->password)) {
                return response()->json(['message' => 'Invalid credentials'], 401);
            }
        }

        Auth::login($user);
        $request->session()->regenerate();

        return response()->json([
            'user' => $user,
            'role' => $user->role,
            'token' => $user->createToken('spa')->plainTextToken,
        ]);
    }

    public function logout(Request $request)
    {
        if ($request->user()) {
            $request->user()->currentAccessToken()?->delete();
        }
        Auth::guard('web')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return response()->json(['message' => 'Logged out successfully']);
    }

    public function me(Request $request)
    {
        return response()->json($request->user());
    }

    public function updateProfile(Request $request)
    {
        $user = $request->user();
        $request->validate([
            'name' => 'required|string|max:255',
        ]);

        $user->name = $request->name;
        $user->save();

        $staff = \App\Models\Staff::where('phone', $user->phone)->first();
        if ($staff) {
            $staff->name = $request->name;
            $staff->save();
        }

        return response()->json([
            'message' => 'Profile updated successfully',
            'user' => $user
        ]);
    }
}
